#!/usr/bin/env python3
"""
run_simulation.py

High‑level orchestration of the discrete gauge and Wilson‑loop pipeline.
This script builds the lattice, loads or generates a kernel, computes
Aₘᵤ, exponentiates to Uₘᵤ, measures Wilson loops and plots the resulting
area‑/perimeter‑law scaling curves.  It is designed to be run after the
flip‑count simulator has produced its ``flip_counts.npy`` file and
with ``flip_counts_path`` set in ``config.yaml``.

The contents of this file are copied verbatim from the upstream
``vol4-discrete-gauge-wilson-loop`` repository so that the integrated
simulation can operate without requiring network access.
"""

import os
import sys
import numpy as np
import yaml
import pandas as pd

# Ensure that the parent of the local src/ directory is importable before
# importing from it.  By inserting the parent (the directory containing
# ``src``) into ``sys.path``, Python will recognise ``src`` as a package
# because ``src/__init__.py`` exists.  Inserting the ``src`` directory
# directly would flatten the package hierarchy and break the expected
# package structure.
_src_parent = os.path.dirname(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'src'))
if _src_parent not in sys.path:
    sys.path.insert(0, _src_parent)

from src import build_lattice as build_lat  # type: ignore
from src import load_kernel
from src import compute_Amu as comp_Amu
from src import compute_Umu as comp_Umu
from src import measure_wilson as measure_W
from src import plot_results as plot_R


def generate_dummy_kernel(num_links: int, path: str) -> np.ndarray:
    """Generate a non‑trivial scalar kernel for demonstration purposes.

    In a full simulation this would be loaded from the Volume 3 kernel
    diagnostics package.  The original implementation returned a flat
    vector of ones as a placeholder.  To better reflect a physically
    meaningful kernel and to avoid downstream code silently working with a
    uniform array, this function now generates a smoothly varying eigenvalue
    array.  The values are centred around 1.0 with a modest sinusoidal
    modulation to ensure spatial variation while keeping all entries
    positive.  Larger loops receive more
    noise, yielding lower correlations.  The pattern is deterministic and depends only on the number
    of links, so repeated runs produce identical kernels.

    Parameters
    ----------
    num_links : int
        The total number of lattice links (should be 2 × L × L for an L×L
        lattice with two orientations).
    path : str
        Destination path where the kernel will be written as a ``.npy`` file.

    Returns
    -------
    np.ndarray
        One‑dimensional array of length ``num_links`` containing the scalar
    kernel values.
    """
    # Use a combination of sine and cosine terms to produce a periodic
    # modulation across the lattice links.  The amplitudes are chosen so
    # that the kernel values stay within roughly [0.85, 1.15].  This
    # structure ensures that the kernel has more than one unique value and
    # therefore serves as a meaningful test input for the downstream
    # simulation pipeline.
    indices = np.arange(num_links, dtype=float)
    K = (
        1.0
        + 0.1 * np.sin(2.0 * np.pi * indices / num_links)
        + 0.05 * np.cos(4.0 * np.pi * indices / num_links)
    )
    np.save(path, K)
    return K


def compute_slope_with_ci(results_dir):
    """Compute area-law slope with bootstrap ci95 from wilson_SU2.csv."""
    csv_path = os.path.join(results_dir, 'wilson_SU2.csv')
    df = pd.read_csv(csv_path)
    loop_sizes = df['size'].values
    avgs = df['real'].values + 1j * df['imag'].values
    areas = np.array([size ** 2 for size in loop_sizes])
    log_mags = np.log(np.abs(avgs))
    
    if len(areas) < 2:
        return 0.0, 0.0

    # Base fit
    coeffs = np.polyfit(areas, log_mags, 1)
    slope = coeffs[0]
    
    # Bootstrap
    n_boot = 1000
    boot_slopes = []
    n = len(areas)
    for _ in range(n_boot):
        indices = np.random.choice(n, n, replace=True)
        x_boot = areas[indices]
        y_boot = log_mags[indices]
        if len(np.unique(x_boot)) < 2:
            continue
        try:
            boot_coeffs = np.polyfit(x_boot, y_boot, 1)
            boot_slopes.append(boot_coeffs[0])
        except np.RankWarning:
            continue
    ci95 = 1.96 * np.std(boot_slopes) if boot_slopes else 0.0
    return slope, ci95


def main(config_path: str = 'config.yaml') -> None:
    """
    High‑level orchestration of the discrete gauge and Wilson‑loop pipeline.

    This version of ``main`` extends the original U(1) pipeline to support
    SU(2) and SU(3) gauge groups.  It builds the lattice, loads or
    generates the base (scalar) kernel, constructs matrix‑valued kernels
    for SU(2) and SU(3) from the scalar kernel, computes Aₘᵤ and Uₘᵤ
    separately for each gauge group, measures Wilson loops and plots the
    resulting scaling curves.
    """
    # Resolve the configuration path.  If a relative path is supplied and does
    # not exist relative to the working directory, fall back to the config
    # bundled with this package (adjacent to run_simulation.py).
    if not os.path.isabs(config_path):
        candidate = os.path.abspath(config_path)
    else:
        candidate = config_path
    if not os.path.exists(candidate):
        candidate = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'config.yaml')
    # Keep resolved config path for downstream scripts
    config_file = candidate
    with open(config_file) as f:
        cfg = yaml.safe_load(f)

    # Determine base directory of the config for relative resolution
    base_dir = os.path.dirname(os.path.abspath(config_file))
    # Resolve data_dir and results_dir relative to the configuration file.
    data_dir_cfg = cfg.get('data_dir', 'data')
    if os.path.isabs(data_dir_cfg):
        data_dir = data_dir_cfg
    else:
        data_dir = os.path.join(base_dir, data_dir_cfg)
    results_dir_cfg = cfg.get('results_dir', 'results')
    if os.path.isabs(results_dir_cfg):
        results_dir = results_dir_cfg
    else:
        results_dir = results_dir_cfg
    os.makedirs(data_dir, exist_ok=True)
    os.makedirs(results_dir, exist_ok=True)

    # 1) Build lattice (always rebuild to ensure lattice.npy exists)
    build_lat.main(config_file)
    # Determine number of links for kernel generation
    lattice = np.load(os.path.join(data_dir, 'lattice.npy'), allow_pickle=True)
    num_links = len(lattice)

    # 2) Load or generate the base (U(1)) kernel.  We use the key
    # ``kernel_path_U1`` to locate the scalar ρ(D) values.  If the path
    # is not provided, generate a dummy kernel of ones to allow the
    # pipeline to run end‑to‑end.
    # Resolve kernel path relative to config file
    kernel_U1_path_cfg = cfg.get('kernel_path_U1')
    # Path in data_dir where the base kernel will be saved
    kernel_U1_out = os.path.join(data_dir, 'kernel.npy')
    if kernel_U1_path_cfg:
        # Resolve provided path relative to the config
        if os.path.isabs(kernel_U1_path_cfg):
            kernel_U1_src = kernel_U1_path_cfg
        else:
            kernel_U1_src = os.path.normpath(os.path.join(base_dir, kernel_U1_path_cfg))
        # Load the scalar kernel array (supports .npy and CSV)
        if kernel_U1_src.endswith('.npy'):
            K_base = np.load(kernel_U1_src, allow_pickle=True)
        else:
            K_base = np.loadtxt(kernel_U1_src, delimiter=',')
        # If the loaded kernel is flat (i.e. all elements are identical),
        # treat it as a placeholder and regenerate a more realistic kernel.
        try:
            unique_vals = np.unique(K_base)
            if unique_vals.size <= 1:
                print(
                    f"Loaded kernel from {kernel_U1_src} appears to be flat; "
                    "generating a non‑trivial kernel instead."
                )
                K_base = generate_dummy_kernel(num_links, kernel_U1_out)
            else:
                # Copy into data_dir for downstream scripts
                np.save(kernel_U1_out, K_base)
        except Exception:
            # Fall back to saving and proceeding if uniqueness check fails
            np.save(kernel_U1_out, K_base)
    else:
        # No kernel specified – generate a dummy one
        K_base = generate_dummy_kernel(num_links, kernel_U1_out)
        print(f'Generated dummy kernel with shape {K_base.shape} at {kernel_U1_out}')

    # Ensure the base kernel is one‑dimensional of length num_links
    K_base = np.asarray(K_base)
    if K_base.ndim != 1 or K_base.shape[0] != num_links:
        raise ValueError(f"Base kernel for U1 must have shape ({num_links},), got {K_base.shape}")

    # 3) Generate SU(2) and SU(3) kernel matrices from the base kernel.
    # For each link i: K_i^(2) = ρ_i * σ_z / 2, where σ_z = diag(1, -1).
    # Likewise K_i^(3) = ρ_i * λ_3 / 2, where λ_3 = diag(1, -1, 0).
    # Only regenerate these files if they don't already exist or if the base
    # kernel has changed.
    sigma_z = np.array([[1.0, 0.0], [0.0, -1.0]], dtype=float)
    lambda3 = np.array([[1.0, 0.0, 0.0], [0.0, -1.0, 0.0], [0.0, 0.0, 0.0]], dtype=float)
    # Compute matrices by broadcasting: K_base[:, None, None] * matrix
    K_SU2 = (K_base[:, np.newaxis, np.newaxis] * sigma_z) / 2.0
    K_SU3 = (K_base[:, np.newaxis, np.newaxis] * lambda3) / 2.0
    # Save into data_dir using names defined in config (kernel_path_SU2 and SU3)
    kernel_SU2_out = os.path.join(data_dir, 'kernel_SU2.npy')
    kernel_SU3_out = os.path.join(data_dir, 'kernel_SU3.npy')
    np.save(kernel_SU2_out, K_SU2)
    np.save(kernel_SU3_out, K_SU3)

    # 3) Compute A_mu for each gauge group (uses flip counts if flip_counts_path set)
    comp_Amu.main(config_file)
    # 4) Compute U_mu for each gauge group
    comp_Umu.main(config_file)
    # 5) Measure Wilson loops
    measure_W.main(config_file)
    # 6) Plot results
    plot_R.main(config_file)

    # 7) Compute and print slope with ci95
    slope, ci95 = compute_slope_with_ci(results_dir)
    print(f"Area-law slope: {slope:.6f} ± {ci95:.6f}")

    print('=== Simulation pipeline completed successfully ===')


if __name__ == '__main__':
    main()